/*
 * Decompiled with CFR 0.152.
 */
package com.squareup.okhttp.internal.http;

import com.squareup.okhttp.Connection;
import com.squareup.okhttp.internal.AbstractOutputStream;
import com.squareup.okhttp.internal.Util;
import com.squareup.okhttp.internal.http.AbstractHttpInputStream;
import com.squareup.okhttp.internal.http.HttpEngine;
import com.squareup.okhttp.internal.http.RawHeaders;
import com.squareup.okhttp.internal.http.ResponseHeaders;
import com.squareup.okhttp.internal.http.RetryableOutputStream;
import com.squareup.okhttp.internal.http.Transport;
import com.squareup.okhttp.internal.http.UnknownLengthHttpInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.CacheRequest;
import java.net.ProtocolException;
import java.net.Socket;

public final class HttpTransport
implements Transport {
    private static final int DISCARD_STREAM_TIMEOUT_MILLIS = 100;
    public static final int DEFAULT_CHUNK_LENGTH = 1024;
    private final HttpEngine httpEngine;
    private final InputStream socketIn;
    private final OutputStream socketOut;
    private OutputStream requestOut;

    public HttpTransport(HttpEngine httpEngine, OutputStream outputStream, InputStream inputStream) {
        this.httpEngine = httpEngine;
        this.socketOut = outputStream;
        this.requestOut = outputStream;
        this.socketIn = inputStream;
    }

    @Override
    public OutputStream createRequestBody() throws IOException {
        boolean chunked = this.httpEngine.requestHeaders.isChunked();
        if (!chunked && this.httpEngine.policy.getChunkLength() > 0 && this.httpEngine.connection.getHttpMinorVersion() != 0) {
            this.httpEngine.requestHeaders.setChunked();
            chunked = true;
        }
        if (chunked) {
            int chunkLength = this.httpEngine.policy.getChunkLength();
            if (chunkLength == -1) {
                chunkLength = 1024;
            }
            this.writeRequestHeaders();
            return new ChunkedOutputStream(this.requestOut, chunkLength);
        }
        long fixedContentLength = this.httpEngine.policy.getFixedContentLength();
        if (fixedContentLength != -1L) {
            this.httpEngine.requestHeaders.setContentLength(fixedContentLength);
            this.writeRequestHeaders();
            return new FixedLengthOutputStream(this.requestOut, fixedContentLength);
        }
        long contentLength = this.httpEngine.requestHeaders.getContentLength();
        if (contentLength > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Use setFixedLengthStreamingMode() or setChunkedStreamingMode() for requests larger than 2 GiB.");
        }
        if (contentLength != -1L) {
            this.writeRequestHeaders();
            return new RetryableOutputStream((int)contentLength);
        }
        return new RetryableOutputStream();
    }

    @Override
    public void flushRequest() throws IOException {
        this.requestOut.flush();
        this.requestOut = this.socketOut;
    }

    @Override
    public void writeRequestBody(RetryableOutputStream requestBody) throws IOException {
        requestBody.writeToSocket(this.requestOut);
    }

    @Override
    public void writeRequestHeaders() throws IOException {
        this.httpEngine.writingRequestHeaders();
        RawHeaders headersToSend = this.httpEngine.requestHeaders.getHeaders();
        byte[] bytes = headersToSend.toBytes();
        this.requestOut.write(bytes);
    }

    @Override
    public ResponseHeaders readResponseHeaders() throws IOException {
        RawHeaders rawHeaders = RawHeaders.fromBytes(this.socketIn);
        this.httpEngine.connection.setHttpMinorVersion(rawHeaders.getHttpMinorVersion());
        this.httpEngine.receiveHeaders(rawHeaders);
        ResponseHeaders headers = new ResponseHeaders(this.httpEngine.uri, rawHeaders);
        headers.setTransport("http/1.1");
        return headers;
    }

    @Override
    public boolean makeReusable(boolean streamCanceled, OutputStream requestBodyOut, InputStream responseBodyIn) {
        if (streamCanceled) {
            return false;
        }
        if (requestBodyOut != null && !((AbstractOutputStream)requestBodyOut).isClosed()) {
            return false;
        }
        if (this.httpEngine.requestHeaders.hasConnectionClose()) {
            return false;
        }
        if (this.httpEngine.responseHeaders != null && this.httpEngine.responseHeaders.hasConnectionClose()) {
            return false;
        }
        if (responseBodyIn instanceof UnknownLengthHttpInputStream) {
            return false;
        }
        if (responseBodyIn != null) {
            return HttpTransport.discardStream(this.httpEngine, responseBodyIn);
        }
        return true;
    }

    private static boolean discardStream(HttpEngine httpEngine, InputStream responseBodyIn) {
        Connection connection = httpEngine.connection;
        if (connection == null) {
            return false;
        }
        Socket socket = connection.getSocket();
        if (socket == null) {
            return false;
        }
        int socketTimeout = socket.getSoTimeout();
        socket.setSoTimeout(100);
        try {
            Util.skipAll(responseBodyIn);
        }
        catch (Throwable throwable) {
            try {
                socket.setSoTimeout(socketTimeout);
                throw throwable;
            }
            catch (IOException e) {
                return false;
            }
        }
        socket.setSoTimeout(socketTimeout);
        return true;
    }

    @Override
    public InputStream getTransferStream(CacheRequest cacheRequest) throws IOException {
        if (!this.httpEngine.hasResponseBody()) {
            return new FixedLengthInputStream(this.socketIn, cacheRequest, this.httpEngine, 0L);
        }
        if (this.httpEngine.responseHeaders.isChunked()) {
            return new ChunkedInputStream(this.socketIn, cacheRequest, this);
        }
        if (this.httpEngine.responseHeaders.getContentLength() != -1L) {
            return new FixedLengthInputStream(this.socketIn, cacheRequest, this.httpEngine, this.httpEngine.responseHeaders.getContentLength());
        }
        return new UnknownLengthHttpInputStream(this.socketIn, cacheRequest, this.httpEngine);
    }

    private static class ChunkedInputStream
    extends AbstractHttpInputStream {
        private static final int NO_CHUNK_YET = -1;
        private final HttpTransport transport;
        private int bytesRemainingInChunk = -1;
        private boolean hasMoreChunks = true;

        ChunkedInputStream(InputStream is, CacheRequest cacheRequest, HttpTransport transport) throws IOException {
            super(is, transport.httpEngine, cacheRequest);
            this.transport = transport;
        }

        @Override
        public int read(byte[] buffer, int offset, int count) throws IOException {
            int read;
            Util.checkOffsetAndCount(buffer.length, offset, count);
            this.checkNotClosed();
            if (!this.hasMoreChunks) {
                return -1;
            }
            if (this.bytesRemainingInChunk == 0 || this.bytesRemainingInChunk == -1) {
                this.readChunkSize();
                if (!this.hasMoreChunks) {
                    return -1;
                }
            }
            if ((read = this.in.read(buffer, offset, Math.min(count, this.bytesRemainingInChunk))) == -1) {
                this.unexpectedEndOfInput();
                throw new IOException("unexpected end of stream");
            }
            this.bytesRemainingInChunk -= read;
            this.cacheWrite(buffer, offset, read);
            return read;
        }

        private void readChunkSize() throws IOException {
            String chunkSizeString;
            int index;
            if (this.bytesRemainingInChunk != -1) {
                Util.readAsciiLine(this.in);
            }
            if ((index = (chunkSizeString = Util.readAsciiLine(this.in)).indexOf(";")) != -1) {
                chunkSizeString = chunkSizeString.substring(0, index);
            }
            try {
                this.bytesRemainingInChunk = Integer.parseInt(chunkSizeString.trim(), 16);
            }
            catch (NumberFormatException e) {
                throw new ProtocolException("Expected a hex chunk size but was " + chunkSizeString);
            }
            if (this.bytesRemainingInChunk == 0) {
                this.hasMoreChunks = false;
                RawHeaders rawResponseHeaders = this.httpEngine.responseHeaders.getHeaders();
                RawHeaders.readHeaders(this.transport.socketIn, rawResponseHeaders);
                this.httpEngine.receiveHeaders(rawResponseHeaders);
                this.endOfInput();
            }
        }

        @Override
        public int available() throws IOException {
            this.checkNotClosed();
            if (!this.hasMoreChunks || this.bytesRemainingInChunk == -1) {
                return 0;
            }
            return Math.min(this.in.available(), this.bytesRemainingInChunk);
        }

        @Override
        public void close() throws IOException {
            if (this.closed) {
                return;
            }
            if (this.hasMoreChunks && !HttpTransport.discardStream(this.httpEngine, this)) {
                this.unexpectedEndOfInput();
            }
            this.closed = true;
        }
    }

    private static final class ChunkedOutputStream
    extends AbstractOutputStream {
        private static final byte[] CRLF = new byte[]{13, 10};
        private static final byte[] HEX_DIGITS = new byte[]{48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 97, 98, 99, 100, 101, 102};
        private static final byte[] FINAL_CHUNK = new byte[]{48, 13, 10, 13, 10};
        private final byte[] hex;
        private final OutputStream socketOut;
        private final int maxChunkLength;
        private final ByteArrayOutputStream bufferedChunk;

        private ChunkedOutputStream(OutputStream socketOut, int maxChunkLength) {
            byte[] byArray = new byte[10];
            byArray[8] = 13;
            byArray[9] = 10;
            this.hex = byArray;
            this.socketOut = socketOut;
            this.maxChunkLength = Math.max(1, this.dataLength(maxChunkLength));
            this.bufferedChunk = new ByteArrayOutputStream(maxChunkLength);
        }

        private int dataLength(int dataPlusHeaderLength) {
            int headerLength = 4;
            int i = dataPlusHeaderLength - headerLength;
            while (i > 0) {
                ++headerLength;
                i >>= 4;
            }
            return dataPlusHeaderLength - headerLength;
        }

        @Override
        public synchronized void write(byte[] buffer, int offset, int count) throws IOException {
            this.checkNotClosed();
            Util.checkOffsetAndCount(buffer.length, offset, count);
            while (count > 0) {
                int numBytesWritten;
                if (this.bufferedChunk.size() > 0 || count < this.maxChunkLength) {
                    numBytesWritten = Math.min(count, this.maxChunkLength - this.bufferedChunk.size());
                    this.bufferedChunk.write(buffer, offset, numBytesWritten);
                    if (this.bufferedChunk.size() == this.maxChunkLength) {
                        this.writeBufferedChunkToSocket();
                    }
                } else {
                    numBytesWritten = this.maxChunkLength;
                    this.writeHex(numBytesWritten);
                    this.socketOut.write(buffer, offset, numBytesWritten);
                    this.socketOut.write(CRLF);
                }
                offset += numBytesWritten;
                count -= numBytesWritten;
            }
        }

        private void writeHex(int i) throws IOException {
            int cursor = 8;
            do {
                this.hex[--cursor] = HEX_DIGITS[i & 0xF];
            } while ((i >>>= 4) != 0);
            this.socketOut.write(this.hex, cursor, this.hex.length - cursor);
        }

        @Override
        public synchronized void flush() throws IOException {
            if (this.closed) {
                return;
            }
            this.writeBufferedChunkToSocket();
            this.socketOut.flush();
        }

        @Override
        public synchronized void close() throws IOException {
            if (this.closed) {
                return;
            }
            this.closed = true;
            this.writeBufferedChunkToSocket();
            this.socketOut.write(FINAL_CHUNK);
        }

        private void writeBufferedChunkToSocket() throws IOException {
            int size = this.bufferedChunk.size();
            if (size <= 0) {
                return;
            }
            this.writeHex(size);
            this.bufferedChunk.writeTo(this.socketOut);
            this.bufferedChunk.reset();
            this.socketOut.write(CRLF);
        }
    }

    private static class FixedLengthInputStream
    extends AbstractHttpInputStream {
        private long bytesRemaining;

        public FixedLengthInputStream(InputStream is, CacheRequest cacheRequest, HttpEngine httpEngine, long length) throws IOException {
            super(is, httpEngine, cacheRequest);
            this.bytesRemaining = length;
            if (this.bytesRemaining == 0L) {
                this.endOfInput();
            }
        }

        @Override
        public int read(byte[] buffer, int offset, int count) throws IOException {
            Util.checkOffsetAndCount(buffer.length, offset, count);
            this.checkNotClosed();
            if (this.bytesRemaining == 0L) {
                return -1;
            }
            int read = this.in.read(buffer, offset, (int)Math.min((long)count, this.bytesRemaining));
            if (read == -1) {
                this.unexpectedEndOfInput();
                throw new ProtocolException("unexpected end of stream");
            }
            this.bytesRemaining -= (long)read;
            this.cacheWrite(buffer, offset, read);
            if (this.bytesRemaining == 0L) {
                this.endOfInput();
            }
            return read;
        }

        @Override
        public int available() throws IOException {
            this.checkNotClosed();
            return this.bytesRemaining == 0L ? 0 : (int)Math.min((long)this.in.available(), this.bytesRemaining);
        }

        @Override
        public void close() throws IOException {
            if (this.closed) {
                return;
            }
            if (this.bytesRemaining != 0L && !HttpTransport.discardStream(this.httpEngine, this)) {
                this.unexpectedEndOfInput();
            }
            this.closed = true;
        }
    }

    private static final class FixedLengthOutputStream
    extends AbstractOutputStream {
        private final OutputStream socketOut;
        private long bytesRemaining;

        private FixedLengthOutputStream(OutputStream socketOut, long bytesRemaining) {
            this.socketOut = socketOut;
            this.bytesRemaining = bytesRemaining;
        }

        @Override
        public void write(byte[] buffer, int offset, int count) throws IOException {
            this.checkNotClosed();
            Util.checkOffsetAndCount(buffer.length, offset, count);
            if ((long)count > this.bytesRemaining) {
                throw new ProtocolException("expected " + this.bytesRemaining + " bytes but received " + count);
            }
            this.socketOut.write(buffer, offset, count);
            this.bytesRemaining -= (long)count;
        }

        @Override
        public void flush() throws IOException {
            if (this.closed) {
                return;
            }
            this.socketOut.flush();
        }

        @Override
        public void close() throws IOException {
            if (this.closed) {
                return;
            }
            this.closed = true;
            if (this.bytesRemaining > 0L) {
                throw new ProtocolException("unexpected end of stream");
            }
        }
    }
}

